<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Models;

use R;
use Exception;
use Throwable;
use RedBeanPHP\OODBBean;
use Erlage\Photogram\System;
use Erlage\Photogram\Session;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\AdminSession;
use Erlage\Photogram\SystemLogger;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Tables\AbstractTable;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Exceptions\ModelDataException;
use Erlage\Photogram\Reflection\TraitReflectionAbstractClassName;

abstract class AbstractModel
{
    /*
    |--------------------------------------------------------------------------
    | reflection helper
    |--------------------------------------------------------------------------
    */

    use TraitReflectionAbstractClassName;

    /*
    |--------------------------------------------------------------------------
    | serializable map data
    |--------------------------------------------------------------------------
    */

    /**
     * @var bool
     */
    private $isModel = false;

    /**
     * @var string
     */
    private $lastError = 'Model data is not ready';

    /**
     * @var OODBBean
     */
    private $bean;

    /*
    |--------------------------------------------------------------------------
    | serializable map data
    |--------------------------------------------------------------------------
    */

    /**
     * @var array
     */
    private $publicPropertiesMap = array();

    /**
     * @var array
     */
    private $privatePropertiesMap = array();

    /**
     * @var array
     */
    private $protectedPropertiesMap = array();

    /*
    |--------------------------------------------------------------------------
    | factories
    |--------------------------------------------------------------------------
    */

    /**
     * @throws ModelDataException 
     * @return static
     */
    final public static function createFromData(array $data)
    {
        return static::wrapModelCreation(function (self $model) use ($data)
        {
            /*
            |--------------------------------------------------------------------------
            | create bean
            |--------------------------------------------------------------------------
            */

            $model -> bean = R::dispense($model -> getTableClassName()::getTableName());

            /*
            |--------------------------------------------------------------------------
            | prepare serializable data for each field
            |--------------------------------------------------------------------------
            */

            foreach ($model -> getTableClassName()::allAttributes() as $databaseAttribute => $attributeAsCamelCase)
            {
                // primary key doesn't exists yet so we need to skip this field
                if ($model -> getTableClassName()::getPrimaryAttribute() == $databaseAttribute)
                {
                    continue;
                }

                // this will invalidate the model if data is not valid
                $model -> updateFieldValue($databaseAttribute, $data[$databaseAttribute]);
            }
        });
    }

    /**
     * [utility method] finds record using primary attribute's value i.e id
     * if record doesn't exists, it throws a exception. for no exceptions
     * use the other method, findFromId_noException()
     * 
     * @see self::findFromId_noException()
     * 
     * @return static 
     * @throws ModelDataException 
     */
    final public static function findFromId_throwException(string $id)
    {
        return self::createFromUntouchedBean_throwException(R::load((new static()) -> getTableClassName()::getTableName(), $id));
    }

    /**
     * finds record using primary attribute's value i.e id
     * if record doesn't exists it returns a dummy model. caller must checks
     * model integrity using isModel call before using the returned result
     * 
     * @return AbstractModel|static|void 
     */
    final public static function findFromId_noException(string $id)
    {
        try
        {
            return self::findFromId_throwException($id);
        }
        catch (Throwable $e)
        {
            return new static();
        }
    }

    /**
     * @return static 
     * @throws Exception 
     */
    final public static function createFromUntouchedBean_noException($finalBean)
    {
        try
        {
            return self::createFromUntouchedBean($finalBean);
        }
        catch (Throwable $e)
        {
            SystemLogger::serverException($e);

            return new static();
        }
    }

    /**
     * @return static 
     * @throws ModelDataException 
     */
    final public static function createFromUntouchedBean_throwException($finalBean)
    {
        return self::createFromUntouchedBean($finalBean);
    }

    /*
    |--------------------------------------------------------------------------
    | Note! We could have all our properties as public. The reason for using 
    | getters is that static checking doesn't work well for undefined properties
    | because of dynamic properties in PHP(__set & __get methods)
    |--------------------------------------------------------------------------
    */

    abstract public function getId(): string;

    final public function isModel(): bool
    {
        return $this -> isModel;
    }

    final public function isNotModel(): bool
    {
        return ! $this -> isModel;
    }

    final public function getBean(): OODBBean
    {
        return $this -> bean;
    }

    final public function getLastError(): string
    {
        return $this -> lastError;
    }

    final public function getDataMap(): array
    {
        if (AdminSession::instance() -> isAuthenticated())
        {
            return $this -> privatePropertiesMap;
        }

        if (Session::instance() -> isAuthenticated())
        {
            if (Session::instance() -> getUserModel() -> getId() == $this -> bean[$this -> getTableClassName()::getOwnerAttribute()])
            {
                return $this -> privatePropertiesMap;
            }
        }

        return $this -> publicPropertiesMap;
    }

    final public function getPrivateMap(): array
    {
        return $this -> privatePropertiesMap;
    }

    final public function getProtectedMap(): array
    {
        return $this -> protectedPropertiesMap;
    }

    /*
    |--------------------------------------------------------------------------
    | field update logic
    |--------------------------------------------------------------------------
    */

    public function update(array $fields = array()): void
    {
        // always update stamp
        $fields[$this -> getTableClassName()::getLastUpdateStampAttribute()] = System::isoDateTime();

        foreach ($fields as $field => $value)
        {
            $this -> updateFieldValue($field, $value);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | saving method
    |--------------------------------------------------------------------------
    */

    public function save(string $exceptionClass = null): int
    {
        if (null == $exceptionClass)
        {
            $exceptionClass = RequestException::erlClass();
        }

        /*
        |--------------------------------------------------------------------------
        | if something has went wrong, don't save the model
        |--------------------------------------------------------------------------
        */

        if ($this -> isNotModel())
        {
            throw new $exceptionClass($this -> getLastError());
        }

        return $this -> presistChanges();
    }

    /**
     * Delete model from database. 
     * 
     * if overrode, overiding method must call super method to delete the model itself.
     * 
     * for now, we're cascading delete to child models, so a top-level delete is hell lot
     * of work. we'll introduce a solution to this problem along with support for MongoDB
     * 
     */
    public function delete(): bool
    {
        if ($this -> isNotModel())
        {
            throw new RequestException($this -> getLastError());
        }

        (new Query())
            -> from($this -> getTableClassName()::getTableName())
            -> where($this -> getTableClassName()::getPrimaryAttribute(), $this -> getId())
            -> delete();

        return true;
    }

    /*
    |--------------------------------------------------------------------------
    | abstract methods
    |--------------------------------------------------------------------------
    */

    /**
     * @return AbstractTable
     * @class-string | Access to static methods/properties only
     */
    abstract protected function getTableClassName(): string;

    abstract protected function getStampLastUpdate(): string;

    /*
    |--------------------------------------------------------------------------
    | field processors
    |--------------------------------------------------------------------------
    */

    /**
     * 
     * @param mixed $value 
     * @return mixed 
     */
    protected function transformField(string $field, $value)
    {
        $transformers = $this -> getTableClassName()::transformers();

        /*
        |--------------------------------------------------------------------------
        | if transformer exists, then transform the value
        |--------------------------------------------------------------------------
        */

        if (isset($transformers[$field]))
        {
            return ($transformers[$field])($value);
        }

        return $value;
    }

    /**
     * @param mixed $value 
     */
    protected function serializeField(string $field, $value): string
    {
        $serializers = $this -> getTableClassName()::serializers();

        /*
        |--------------------------------------------------------------------------
        | if serializer exists, then serialize the value
        |--------------------------------------------------------------------------
        */

        if (isset($serializers[$field]))
        {
            return ($serializers[$field])($value);
        }

        return $value;
    }

    /**
     * @param mixed $value 
     * @return mixed 
     */
    protected function deserializeField(string $field, $value)
    {
        $deSerializers = $this -> getTableClassName()::deSerializers();

        /*
        |--------------------------------------------------------------------------
        | if deserializer exists, then deserialize the value
        |--------------------------------------------------------------------------
        */

        if (isset($deSerializers[$field]))
        {
            return ($deSerializers[$field])($value);
        }

        return $value;
    }

    /**
     * @param mixed $value 
     */
    protected function validateField(string $field, $value): string
    {
        $validators = $this -> getTableClassName()::validators();

        /*
        |--------------------------------------------------------------------------
        | if post processor exists, then post process it
        |--------------------------------------------------------------------------
        */

        if (isset($validators[$field]))
        {
            return ($validators[$field])($value);
        }

        return SystemConstants::OK;
    }

    /**
     * sets attribute value. same like setFieldValue, but this method has to be used
     * for all values that are coming from untrusted source. typically everything except
     * database are untrusted and use this method to set attribute value for those type
     * of sources. this method will make sure that internal system run all validation,
     * transformation and serialization tasks before adding/updating anything to database
     * or even model object.
     * 
     * @param mixed $value 
     */
    protected function updateFieldValue(string $field, $value): void
    {
        $this -> alterFieldValue($field, $value);
    }

    /**
     * sets attribute value. use this method to set attribute value to a trusted 
     * value. typically use it only for valuse that are coming directly from
     * database. using this method will alter attribute value while preventing any
     * additional validation, serialization and or transformation tasks from running.
     * 
     * @param mixed $value 
     */
    protected function setFieldValue(string $field, $value): void
    {
        $this -> alterFieldValue($field, $value, true);
    }

    /**
     * change model's attribute's value.
     * it also runs validation, transformations, serializations for attribute if they
     * are defined in table definition. 
     * 
     * @param mixed $value 
     */
    protected function alterFieldValue(string $property, $value, bool $fetchedFromDatabase = false): void
    {
        /*
        |--------------------------------------------------------------------------
        | set field only if model is valid
        |--------------------------------------------------------------------------
        */

        if ($this -> isNotModel())
        {
            return;
        }

        /*
        |--------------------------------------------------------------------------
        | transform values that are [not available] in user's input
        |--------------------------------------------------------------------------
        */

        if ( ! $fetchedFromDatabase && \is_string($value) && SystemConstants::NA == $value)
        {
            $value = '';
        }

        /*
        |--------------------------------------------------------------------------
        | validate field data, if not fetched from database
        |--------------------------------------------------------------------------
        */

        if ( ! $fetchedFromDatabase)
        {
            $validationResults = $this -> validateField($property, $value);

            if (SystemConstants::OK != $validationResults)
            {
                $this -> isModel = false;

                $this -> lastError = $validationResults;

                return;
            }
        }

        /*
        |--------------------------------------------------------------------------
        | transform data
        |--------------------------------------------------------------------------
        */

        if ( ! $fetchedFromDatabase)
        {
            // transform value so we can push it to database later
            $transformedValue = $this -> transformField($property, $value);
        }

        /*
        |--------------------------------------------------------------------------
        | serialize the value
        |--------------------------------------------------------------------------
        */

        if ($fetchedFromDatabase)
        {
            // no-need, coming directly from database
            $serializedValue = $value;
        }
        else
        {
            // serialized transformed & possibled updated value
            $serializedValue = $this -> serializeField($property, $transformedValue);
        }

        /*
        |--------------------------------------------------------------------------
        | de-serialize data
        |--------------------------------------------------------------------------
        */

        $deserializedValue = $this -> deserializeField($property, $serializedValue);

        /*
        |--------------------------------------------------------------------------
        | update bean, use serialized value
        |--------------------------------------------------------------------------
        */

        $this -> bean -> {$property} = $serializedValue;

        /*
        |--------------------------------------------------------------------------
        | update object {field} property, use deserialzied value
        |--------------------------------------------------------------------------
        */

        $this -> { $this -> getTableClassName()::allAttributes()[$property] }
        = $deserializedValue;

        /*
        |--------------------------------------------------------------------------
        | prepare attribute's types/category
        |--------------------------------------------------------------------------
        */

        $isPrivateAttribute = (\array_key_exists($property, $this -> getTableClassName()::privateAttributes()));

        $isProtectedAttribute = (\array_key_exists($property, $this -> getTableClassName()::protectedAttributes()));

        /**
         * 1. Public values
         *      - are sent to all users without in their raw representation
         * 
         * 2. Private values
         *      - are sent to client only if client is the owner of those values
         * 
         * 3. Protected values
         *      - are never sent to client and they're available only at server
         */

        /*
        |--------------------------------------------------------------------------
        | update public map
        |--------------------------------------------------------------------------
        */

        if ( ! $isPrivateAttribute && ! $isProtectedAttribute)
        {
            $this -> publicPropertiesMap[$property] = $deserializedValue;
        }

        /*
        |--------------------------------------------------------------------------
        | update private map
        |--------------------------------------------------------------------------
        */

        if ( ! $isProtectedAttribute)
        {
            $this -> privatePropertiesMap[$property] = $deserializedValue;
        }

        /*
        |--------------------------------------------------------------------------
        | update protected map
        |--------------------------------------------------------------------------
        */

        $this -> protectedPropertiesMap[$property] = $deserializedValue;
    }

    /**
     * Tries to store model into database, if model doesn't exists it will create it
     * and insert it in the database.
     * 
     * @return Int primary attribute value of record(from database) 
     */
    protected function presistChanges(): int
    {
        /*
        |--------------------------------------------------------------------------
        | make sure model data is valid
        |--------------------------------------------------------------------------
        */

        $primaryAttributeValue = R::store($this -> bean);

        $this -> setFieldValue($this -> getTableClassName()::getPrimaryAttribute(), (string) $primaryAttributeValue);

        return $primaryAttributeValue;
    }

    /**
     * @return static 
     * @throws ModelDataException 
     */
    private static function createFromUntouchedBean($finalBean)
    {
        return static::wrapModelCreation(function (self $model) use ($finalBean)
        {
            try
            {
                /*
                |--------------------------------------------------------------------------
                | if bean doesn't exists in database, null when bean is feeded by R::find()
                |--------------------------------------------------------------------------
                */

                if (null == $finalBean)
                {
                    throw new ModelDataException('ModelDataException: Bean not found');
                }

                /*
                |--------------------------------------------------------------------------
                | if bean doesn't exists in database, 0 when feeded by R::load()
                |--------------------------------------------------------------------------
                */

                if (0 == $finalBean[$model -> getTableClassName()::getPrimaryAttribute()])
                {
                    throw new ModelDataException('ModelDataException: Corrupted data');
                }

                /*
                |--------------------------------------------------------------------------
                | set bean
                |--------------------------------------------------------------------------
                */

                $model -> bean = $finalBean;
            }
            catch (ModelDataException $e)
            {
                throw $e;
            }
            catch (Throwable $e)
            {
                throw new ModelDataException('Something went wrong while creating model');
            }

            /*
            |--------------------------------------------------------------------------
            | prepare serializable data for each field
            |--------------------------------------------------------------------------
            */

            foreach ($model -> getTableClassName()::allAttributes() as $databaseAttribute => $attributeAsCamelCase)
            {
                $model -> setFieldValue($databaseAttribute, $model -> bean -> {$databaseAttribute});
            }
        });
    }

    /**
     * This methods wraps the model creating inside a try-catch block 
     * 
     * @return static 
     * @throws ModelDataException 
     */
    private static function wrapModelCreation(callable $procedure)
    {
        $model = new static();

        try
        {
            /*
            |--------------------------------------------------------------------------
            | start fresh
            |--------------------------------------------------------------------------
            */

            $model -> isModel = true;

            $procedure($model);
        }
        catch (ModelDataException $e)
        {
            $model -> isModel = false;

            throw $e;
        }
        catch (Exception $e)
        {
            /*
            |--------------------------------------------------------------------------
            | if anything goes wrong, that means data isn't correct
            |--------------------------------------------------------------------------
            */

            $model -> isModel = false;

            $model -> lastError = $e -> getMessage();
        }

        return $model;
    }
}
